/**
 * \file sdc_op_wrap_unwrap.c
 *
 * \brief Functions for wrap and unwrap
 * Please note : the implementation is split into different operations
 * instead of splitting it in common, convenience and advanced functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc_op_common.h>
#include <sdc_op_adv.h>
#include <sdc_op_conv.h>
#include <sdc_random.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */

static const char *sdc_wrap_unwrap_alg_names[] = {
    [SDC_WRAP_ALG_AES] = "AES"
};

static const char *sdc_wrap_unwrap_blk_names[] = {
    [SDC_WRAP_BLK_CCM] = "CCM",
    [SDC_WRAP_BLK_GCM] = "GCM"
};

/* Functions */

/**
 * \brief Checks common to wrap and unwrap
 *
 * Check session, type and in and out data
 * Provide \ref sdc_wrap_unwrap_desc_t of type
 */
static sdc_error_t sdc_wrap_unwrap_common_checks_defaults(
    sdc_error_t error_init,
    sdc_session_t *session,
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *internal_desc,
    const uint8_t *in_data, const size_t in_len,
    uint8_t **out_data, size_t *out_len, const sdc_error_t out_err)
{

    sdc_error_t err = error_init;

    if (SDC_OK != sdc_intern_check_init_data_output_buffer(out_data, out_len))
        err = out_err;

    if (SDC_OK != sdc_intern_check_data_input_buffer(in_data, in_len))
        err = SDC_IN_DATA_INVALID;

    if (!type)
        err = SDC_ALG_MODE_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_desc_fill (session, type, internal_desc, in_len);

    return err;
}

/**
 * \brief call min max checks for in data, iv and tag
 */
static sdc_error_t sdc_wrap_unwrap_common_check_iv_tag_in_length(
    const sdc_wrap_unwrap_desc_t *desc,
    bool is_wrap,
    size_t in_len,
    size_t iv_len,
    size_t tag_len
    )
{
    sdc_error_t err;

    err = sdc_intern_inout_input_check_min_max_align(in_len,
                                                           &(desc->data),
                                                           /* use plain when wrap else cipher spec */
                                                           is_wrap);
    if (err != SDC_OK)
        return err;

    if (SDC_OK != sdc_intern_range_min_max_mod_check(iv_len, &(desc->iv)))
        return SDC_IV_INVALID;

    if (SDC_OK != sdc_intern_range_min_max_mod_check(tag_len, &(desc->tag)))
        return SDC_TAG_DATA_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_session_type_desc_validation(sdc_session_t *session,
                                              const sdc_wrap_unwrap_type_t *type,
                                              const sdc_wrap_unwrap_desc_t *desc)
{
    /* verify inputs */
    if (!session)
        return SDC_SESSION_INVALID;
    if(!type)
        return SDC_ALG_MODE_INVALID;
    if(!desc)
        return SDC_INVALID_PARAMETER;

    return SDC_OK;
}

/* check output parameters */
static sdc_error_t sdc_wrap_unwrap_check_output (uint8_t *data, size_t *len)
{

    if ((data == NULL) || (len == NULL))
        return SDC_OUT_DATA_INVALID;

    return SDC_OK;
}

/**
 * \brief call min max checks for in data and iv
 */
static sdc_error_t sdc_wrap_unwrap_common_check_iv(const sdc_wrap_unwrap_desc_t *desc, size_t iv_len )
{
    if (sdc_intern_range_min_max_mod_check(iv_len, &(desc->iv)) != SDC_OK)
        return SDC_IV_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_wrap_unwrap_common_check_tag(const sdc_wrap_unwrap_desc_t *desc, size_t tag_len)
{
    if (SDC_OK != sdc_intern_range_min_max_mod_check(tag_len, &(desc->tag)))
        return SDC_TAG_DATA_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_wrap_get_max_len(sdc_session_t *session,
                                        const sdc_wrap_unwrap_type_t *type,
                                        const sdc_wrap_unwrap_desc_t *desc,
                                        const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err = SDC_OK;

    if (out_len_max == NULL)
        err = SDC_OUT_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_session_type_desc_validation(session, type, desc);

    if (err == SDC_OK)
        err = sdc_arch_wrap_get_out_len(session, type, desc, in_len, out_len_max);

    return err;
}

static sdc_error_t sdc_unwrap_get_max_len(sdc_session_t *session,
                                          const sdc_wrap_unwrap_type_t *type,
                                          const sdc_wrap_unwrap_desc_t *desc,
                                          const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err = SDC_OK;

    if (out_len_max == NULL)
        err = SDC_OUT_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_session_type_desc_validation(session, type, desc);

    if (err == SDC_OK)
        err = sdc_arch_unwrap_get_max_out_len(session, type, desc, in_len, out_len_max);

    return err;
}

sdc_error_t sdc_wrap_get_overall_len(sdc_session_t *session,
                                     const sdc_wrap_unwrap_type_t *type,
                                     const sdc_wrap_unwrap_desc_t *desc,
                                     const size_t in_len, size_t *out_len_max)
{
    return sdc_wrap_get_max_len(session, type, desc, in_len, out_len_max);
}

sdc_error_t sdc_unwrap_get_overall_len(sdc_session_t *session,
                                       const sdc_wrap_unwrap_type_t *type,
                                       const sdc_wrap_unwrap_desc_t *desc,
                                       const size_t in_len, size_t *out_len_max)
{
    return sdc_unwrap_get_max_len(session, type, desc, in_len, out_len_max);
}

static sdc_error_t sdc_wrap_unwrap_get_update_len(sdc_session_t *session,
                                                  const sdc_wrap_unwrap_type_t *type,
                                                  const sdc_wrap_unwrap_desc_t *desc,
                                                  const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err = SDC_OK;
    size_t padding_bytes;
    size_t block_len;

    if(out_len_max == NULL)
        err = SDC_OUT_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_wrap_get_max_len(session, type, desc, in_len, out_len_max);

    if(err == SDC_OK) {
        block_len = desc->data.block_len;
        padding_bytes = in_len % block_len;
        padding_bytes = block_len - padding_bytes;
        if(out_len_max)
            *out_len_max+= padding_bytes;
    }

    return err;
}

sdc_error_t sdc_wrap_get_update_len(sdc_session_t *session,
                                    const sdc_wrap_unwrap_type_t *type,
                                    const sdc_wrap_unwrap_desc_t *desc,
                                    const size_t in_len, size_t *out_len_max)
{
    return sdc_wrap_unwrap_get_update_len(session, type, desc, in_len, out_len_max);
}

sdc_error_t sdc_unwrap_get_update_len(sdc_session_t *session,
                                      const sdc_wrap_unwrap_type_t *type,
                                      const sdc_wrap_unwrap_desc_t *desc,
                                      const size_t in_len, size_t *out_len_max)
{
    return sdc_wrap_unwrap_get_update_len(session, type, desc, in_len, out_len_max);
}

static sdc_error_t sdc_get_wrap_unwrap_tag_length(const sdc_wrap_unwrap_desc_t *desc, size_t *tag_len)
{
    if(tag_len !=  NULL) {
        if(*tag_len == SDC_TAG_USE_DEFAULT)
            *tag_len = desc->tag.dflt;

        return sdc_wrap_unwrap_common_check_tag(desc, *tag_len);
    }

    return SDC_TAG_DATA_INVALID;
}

static sdc_error_t sdc_common_wrapunwrap_init(sdc_session_t *session,
                                              const sdc_wrap_unwrap_type_t *type,
                                              const sdc_wrap_unwrap_desc_t *desc,
                                              const size_t total_inlen,
                                              const size_t total_aadlen,
                                              size_t tag_len,
                                              const uint8_t *iv, size_t iv_len,
                                              bool is_wrap)
{
    session->unaligned_buffer_filllevel = 0;
    session->inlen_cnt = 0;

    session->iuf_wrap_unwrap.total_inlen = (size_t)total_inlen;
    session->iuf_wrap_unwrap.total_aadlen = (size_t)total_aadlen;
    session->iuf_wrap_unwrap.tag_len = (size_t)tag_len;

    if (!desc->supports_iuf)
        return SDC_OP_NOT_SUPPORTED;

    if (is_wrap)
        return sdc_arch_wrap_init(session, type, desc, iv, iv_len);

    return sdc_arch_unwrap_init(session, type, desc, iv, iv_len);
}

static sdc_error_t sdc_common_wrapunwrap_aad_update(sdc_session_t *session,
                                                    const sdc_wrap_unwrap_type_t *type,
                                                    const sdc_wrap_unwrap_desc_t *desc,
                                                    const uint8_t *aad_data, const size_t aad_len, bool is_wrap)
{
    sdc_error_t err = SDC_OK;

    /* Currently we are directly passing aad_data to architecture dependent functions, but in long run we may need
     * handle padding, alignment... etc here */
    if(is_wrap)
        err = sdc_arch_wrap_aad_update(session, type, desc, aad_data, aad_len);
    else
        err = sdc_arch_unwrap_aad_update(session, type, desc, aad_data, aad_len);

    return err;
}

static sdc_error_t sdc_common_wrapunwrap_update(sdc_session_t *session,
                                                 const sdc_wrap_unwrap_type_t *type,
                                                 const sdc_wrap_unwrap_desc_t *desc,
                                                 const uint8_t *in_data, const size_t in_data_len,
                                                 uint8_t *out_data, size_t *out_data_len,
                                                 bool is_wrap)
{
    sdc_error_t err = SDC_OK;
    size_t block_len = desc->data.block_len;
    size_t chunk_len = desc->data.max_chunk_len;
    bool chunk_len_aligned = desc->data.chunk_len_aligned;
    const uint8_t *next_in_data;
    size_t in_len_remaining;
    uint8_t *next_out_data;
    size_t out_len_remaining;
    size_t out_len_used;
    size_t unaligned_buf_rem;
    uint8_t *unaligned_buf_next;
    size_t used_out_len;
    size_t min_no_proc_len;
    size_t process_in_len;
    size_t remainder;

    if (session->unaligned_buffer_filllevel > block_len)
        err = SDC_INTERNAL_ERROR;

    if(in_data_len > *out_data_len)
        err = SDC_OUT_DATA_INVALID;

    /* Check that processing additional data won't exceed max */
    if (err == SDC_OK)
        err = sdc_intern_inout_input_check_update_wont_exceed_max(
                session->inlen_cnt,
                in_data_len,
                &(desc->data),
                /* use plain when wrap else cipher spec */
                is_wrap);

    /* verify that new input won't exceed the total length specified in init */
    if (err == SDC_OK) {
        if (session->inlen_cnt > session->iuf_wrap_unwrap.total_inlen) {
            /*
             * The functions should only process the data if the max wouldn't be
             * exceeded - so this should not happen.
             * The additional check was done to prevent overflows due to implementation errors.
             */
            err = SDC_INTERNAL_ERROR;
        } else {
            size_t remaining_len = session->iuf_wrap_unwrap.total_inlen - session->inlen_cnt;

            if (in_data_len > remaining_len)
                err = SDC_IN_DATA_INVALID;
        }
    }

    next_in_data = in_data;
    in_len_remaining = in_data_len;
    next_out_data = out_data;
    out_len_remaining = *out_data_len;
    out_len_used = 0;

    /* handle previous unaligned data first */
    if ((err == SDC_OK) && (session->unaligned_buffer_filllevel > 0)) {
        /* if unaligned_buffer_filllevel > 0 we obviously need to alignment */
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;
        if (!chunk_len_aligned) {
            /*
             * This must not happen
             * Probably someone has corrupted the control structures
             */
            err = SDC_INTERNAL_ERROR;
        } else {
            if (in_len_remaining > unaligned_buf_rem)
            {
                unaligned_buf_next = session->unaligned_buffer;
                unaligned_buf_next += session->unaligned_buffer_filllevel;

                /* afterwards we have on block in the buffer */
                memcpy(unaligned_buf_next, next_in_data, unaligned_buf_rem);

                /* process the unaligned part */
                used_out_len = out_len_remaining;
                if (is_wrap) {
                    err = sdc_arch_wrap_update(session, type, desc,
                                               session->unaligned_buffer, block_len,
                                               next_out_data, &used_out_len);
                } else {
                    err = sdc_arch_unwrap_update(session, type, desc,
                                                 session->unaligned_buffer, block_len,
                                                 next_out_data, &used_out_len);
                }
                if (err == SDC_OK) {
                    /* update pointers + lengths for remaining data */
                    next_in_data += unaligned_buf_rem;
                    in_len_remaining -= unaligned_buf_rem;
                    next_out_data += used_out_len;
                    out_len_remaining -= used_out_len;
                    out_len_used += used_out_len;
                    session->unaligned_buffer_filllevel = 0;
                }
            }
            /*
             * else case
             *       i.e. in_len_remaining < unaligned_buf_mem
             * will be handled by "append remaining data to unaligned_buffer"
             *
             * In this case
             *      in_len_remaining is < block_len as well and chunk_len_aligned is true
             *
             * Otherwise there is no way (beside fooling around with internal
             * structs) that data has been added to unaligned_buffer in the
             * previous call.
             *
             * Note: the while loop won't process any data in this case
             */
        }
    }

    /* smaller or equal length need to be handled using unaligned_buffer */
    min_no_proc_len = 0;
    if (chunk_len_aligned)
        min_no_proc_len = block_len - 1;

    while ((err == SDC_OK) && (in_len_remaining > min_no_proc_len)) {
        process_in_len = in_len_remaining;

        if (process_in_len > chunk_len)
            process_in_len = chunk_len;
        if (chunk_len_aligned) {
            /* align */
            remainder = (process_in_len % block_len);
            if (remainder != 0) {
                process_in_len -= remainder;
            }
        }

        /* process the next chunk of data */
        used_out_len = out_len_remaining;
        if (is_wrap) {
            err = sdc_arch_wrap_update(session, type, desc,
                                       next_in_data, process_in_len,
                                       next_out_data, &used_out_len);
        } else {
            err = sdc_arch_unwrap_update(session, type, desc,
                                         next_in_data, process_in_len,
                                         next_out_data, &used_out_len);
        }

        if (err == SDC_OK) {
            /* update pointers + lengths for remaining data */
            next_in_data += process_in_len;
            in_len_remaining -= process_in_len;
            next_out_data += used_out_len;
            out_len_remaining -= used_out_len;
            out_len_used += used_out_len;
        }
    }

    /* append remaining data to unaligned_buffer */
    if ((err == SDC_OK) && (in_len_remaining > 0)) {
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (in_len_remaining > unaligned_buf_rem) {
            /* this must not happen */
            err = SDC_INTERNAL_ERROR;
        } else {
            unaligned_buf_next = session->unaligned_buffer;
            unaligned_buf_next += session->unaligned_buffer_filllevel;

            /* append to the end of the unaligned buffer */
            memcpy(unaligned_buf_next,
                   next_in_data,
                   in_len_remaining);
            session->unaligned_buffer_filllevel += in_len_remaining;

            /* no need to update in_data_remaining */
            in_len_remaining = 0;
        }
    }

    *out_data_len = out_len_used;

    if (err != SDC_OK) {
        /* clear confidential data in case of error */
        sdc_intern_clear_session_confidential(session);
    } else {
        /* add the current input data length */
        session->inlen_cnt += in_data_len;
    }

    return err;
}

static sdc_error_t sdc_common_wrapunwrap_finalize(sdc_session_t *session,
                                                  const sdc_wrap_unwrap_type_t *type,
                                                  const sdc_wrap_unwrap_desc_t *desc,
                                                  uint8_t *out_tag_data, size_t out_tag_len,
                                                  const uint8_t *in_tag_data, size_t in_tag_len,
                                                  uint8_t *out_data, size_t *out_len,
                                                  bool is_wrap)
{
    sdc_error_t err = SDC_OK;
    size_t unaligned_data_len;
    sdc_padding_t padding = desc->data.padding;

    unaligned_data_len = session->unaligned_buffer_filllevel;

    /* Current wrap/unwrap formats don't have padding - so this is not handled
     * In future padding can be implemented here. */
    if ((padding != SDC_PADDING_INTERNAL) && (padding != SDC_PADDING_NO))
        return SDC_INTERNAL_ERROR;

    err = sdc_intern_inout_input_check_min_max_align(
            session->inlen_cnt,
            &(desc->data),
            /* use plain when wrap else cipher spec */
            is_wrap);

    /* verify that overall input length is equal to length specified in init */
    if ((err == SDC_OK) &&
        (session->inlen_cnt != session->iuf_wrap_unwrap.total_inlen))
        err = SDC_IN_DATA_INVALID;

    if (err == SDC_OK) {
        if (is_wrap) {
            err = sdc_arch_wrap_finalize(session, type, desc,
                                         session->unaligned_buffer, unaligned_data_len,
                                         out_tag_data, out_tag_len,
                                         out_data, out_len);

        } else {
            err = sdc_arch_unwrap_finalize(session, type, desc,
                                           session->unaligned_buffer, unaligned_data_len,
                                           in_tag_data, in_tag_len,
                                           out_data, out_len);

        }
    }

    sdc_intern_clear_session_confidential(session);

    return err;
}

/**
 * \brief common implementation of wrap using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized wrap functionality
 */
static sdc_error_t sdc_common_wrap(sdc_session_t *session,
                                   const sdc_wrap_unwrap_type_t *type,
                                   const sdc_wrap_unwrap_desc_t *desc,
                                   const uint8_t *in_data,
                                   const size_t in_data_len,
                                   uint8_t *out_data,
                                   const size_t out_data_len,
                                   const uint8_t *aad_data, const size_t aad_len,
                                   const uint8_t *iv, const size_t iv_len,
                                   uint8_t *tag_out_data, const size_t tag_out_len)
{

    sdc_error_t err = SDC_OK;
    size_t used_out_data;
    uint8_t *next_out_data = out_data;
    size_t remaining_out_data_len = out_data_len;

    err = sdc_common_wrapunwrap_init(session, type, desc,
                                     in_data_len, aad_len, (size_t)tag_out_len,
                                     iv, iv_len, true);
    if (err == SDC_OK) {
        err = sdc_common_wrapunwrap_aad_update(session, type, desc, aad_data, aad_len, true);
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_wrapunwrap_update(session, type, desc,
                                           in_data, in_data_len,
                                           next_out_data, &used_out_data,
                                           true);
        next_out_data += used_out_data;
        remaining_out_data_len -= used_out_data;
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_wrapunwrap_finalize(session, type, desc,
                                             tag_out_data, tag_out_len,
                                             NULL, 0,
                                             next_out_data, &used_out_data,
                                             true);
        /* not needed: next_out_data += used_out_data; */
        remaining_out_data_len -= used_out_data;
    }

    /*
     * for wrap the exact output length is known in advance
     * therefore the complete buffer needs to be used
     */
    if ((err == SDC_OK) && (remaining_out_data_len != 0))
        err = SDC_INTERNAL_ERROR;

    return err;
}

/**
 * \brief common implementation of unwrap using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized unwrap functionality
 */
static sdc_error_t sdc_common_unwrap(sdc_session_t *session,
                                     const sdc_wrap_unwrap_type_t *type,
                                     const sdc_wrap_unwrap_desc_t *desc,
                                     const uint8_t *in_data,
                                     const size_t in_data_len,
                                     uint8_t *out_data,
                                     size_t *out_data_len,
                                     const uint8_t *aad_data, const size_t aad_len,
                                     const uint8_t *tag_in_data, const size_t tag_in_len,
                                     const uint8_t *iv, const size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    size_t used_out_data;
    uint8_t *next_out_data = out_data;
    size_t remaining_out_data_len = *out_data_len;
    size_t sum_out_len = 0;

    err = sdc_common_wrapunwrap_init(session, type, desc,
                                     in_data_len, aad_len, (size_t)tag_in_len,
                                     iv, iv_len, false);
    if (err == SDC_OK) {
        err = sdc_common_wrapunwrap_aad_update(session, type, desc, aad_data, aad_len, false);
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_wrapunwrap_update(session, type, desc,
                                           in_data, in_data_len,
                                           next_out_data, &used_out_data,
                                           false);
        next_out_data += used_out_data;
        remaining_out_data_len -= used_out_data;
        sum_out_len += used_out_data;
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_wrapunwrap_finalize(session, type, desc,
                                             NULL, 0,
                                             tag_in_data, tag_in_len,
                                             next_out_data, &used_out_data,
                                             false);
        /* not needed: next_out_data += used_out_data; */
        /* not needed: remaining_out_data_len -= used_out_data; */
        sum_out_len += used_out_data;
    }

    if (err == SDC_OK)
        *out_data_len = sum_out_len;

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_wrap_selector(sdc_session_t *session,
                                     const sdc_wrap_unwrap_type_t *type,
                                     const sdc_wrap_unwrap_desc_t *desc,
                                     const uint8_t *in_data,
                                     const size_t in_data_len,
                                     uint8_t *out_data,
                                     const size_t out_data_len,
                                     const uint8_t *aad_data, const size_t aad_len,
                                     const uint8_t *iv, const size_t iv_len,
                                     uint8_t *tag_out_data, const size_t tag_out_len)
{

    sdc_error_t err;

    err = sdc_arch_wrap(session,
                        type,
                        desc,
                        in_data,
                        in_data_len,
                        out_data,
                        out_data_len,
                        aad_data, aad_len,
                        iv, iv_len,
                        tag_out_data, tag_out_len);
    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_wrap(session,
                              type,
                              desc,
                              in_data,
                              in_data_len,
                              out_data,
                              out_data_len,
                              aad_data, aad_len,
                              iv, iv_len,
                              tag_out_data, tag_out_len);
    }

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_unwrap_selector(sdc_session_t *session,
                                       const sdc_wrap_unwrap_type_t *type,
                                       const sdc_wrap_unwrap_desc_t *desc,
                                       const uint8_t *in_data,
                                       const size_t in_data_len,
                                       uint8_t *out_data,
                                       size_t *out_data_len,
                                       const uint8_t *aad_data, const size_t aad_len,
                                       const uint8_t *tag_in_data, const size_t tag_in_len,
                                       const uint8_t *iv, const size_t iv_len)
{

    sdc_error_t err;

    err = sdc_arch_unwrap(session,
                          type,
                          desc,
                          in_data,
                          in_data_len,
                          out_data,
                          out_data_len,
                          aad_data, aad_len,
                          tag_in_data, tag_in_len,
                          iv, iv_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_unwrap(session,
                                type,
                                desc,
                                in_data,
                                in_data_len,
                                out_data,
                                out_data_len,
                                aad_data, aad_len,
                                tag_in_data, tag_in_len,
                                iv, iv_len);
    }

    return err;
}


sdc_error_t sdc_wrap(sdc_session_t *session,
                     const sdc_wrap_unwrap_type_t *type,
                     const uint8_t *in_data, const size_t in_len,
                     const uint8_t *aad_data, const size_t aad_len,
                     uint8_t **iv, size_t *iv_len,
                     uint8_t **out_data, size_t *out_len,
                     uint8_t **tag_data, size_t *tag_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_wrap_unwrap_desc_t internal_desc;

    bool explicit_iv;
    uint8_t *internal_iv;
    size_t internal_iv_len;
    uint8_t *internal_tag = NULL;
    size_t internal_tag_len;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;

    /* initialize - in case failing with error we want to return
     * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
     */

    if (SDC_OK != sdc_intern_check_data_input_buffer(aad_data, aad_len))
        err = SDC_AAD_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(tag_data, tag_len, &internal_tag_len, SDC_TAG_USE_DEFAULT, NULL))
        err = SDC_TAG_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv)) {
        err = SDC_IV_INVALID;
    }

    err = sdc_wrap_unwrap_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len,
                                                 out_data, out_len, SDC_OUT_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    if (explicit_iv) {
        internal_iv = *iv;
    } else {
        if (internal_iv_len == SDC_IV_USE_DEFAULT) {
            internal_iv_len = internal_desc.iv.dflt;
        }
        internal_iv = NULL;
        /* there might be formats without IV */
        if (internal_iv_len != 0) {
            err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
        }
    }

    if (internal_tag_len == SDC_TAG_USE_DEFAULT) {
        internal_tag_len = internal_desc.tag.dflt;
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_common_check_iv_tag_in_length(
            &internal_desc,
            true,
            in_len,
            internal_iv_len,
            internal_tag_len);
    }

    if (err == SDC_OK) {
        /* determine the output length */
        err = sdc_arch_wrap_get_out_len(session, type, &internal_desc, in_len, &internal_out_len);
    }

    if (err == SDC_OK) {
        if (internal_out_len) {
            internal_out = malloc(internal_out_len);
            if (!internal_out)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK) {
        if (internal_tag_len != 0) {
            internal_tag = malloc(internal_tag_len);
            if (!internal_tag)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK) {
        err = sdc_wrap_selector(session,
                                type,
                                &internal_desc,
                                in_data,
                                in_len,
                                internal_out,
                                internal_out_len,
                                aad_data, aad_len,
                                internal_iv, internal_iv_len,
                                internal_tag, internal_tag_len);
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
        *tag_data = internal_tag;
        *tag_len = internal_tag_len;
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
        free(internal_out);
        free(internal_tag);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap(sdc_session_t *session,
                       const sdc_wrap_unwrap_type_t *type,
                       const uint8_t *in_data, const size_t in_len,
                       const uint8_t *aad_data, const size_t aad_len,
                       const uint8_t *iv, const size_t iv_len,
                       const uint8_t *tag_data, const size_t tag_len,
                       uint8_t **out_data, size_t *out_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_wrap_unwrap_desc_t internal_desc;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;

    if (SDC_OK != sdc_intern_check_data_input_buffer(aad_data, aad_len))
        err = SDC_AAD_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(tag_data, tag_len, SDC_TAG_USE_DEFAULT))
        err = SDC_TAG_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    err = sdc_wrap_unwrap_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len,
                                                 out_data, out_len, SDC_OUT_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    err = sdc_wrap_unwrap_common_check_iv_tag_in_length(
        &internal_desc,
        false,
        in_len,
        iv_len,
        tag_len);

    if (err == SDC_OK) {
        /* determine the max output length */
        err = sdc_arch_unwrap_get_max_out_len(
            session, type, &internal_desc,
            in_len,
            &internal_out_len);
    }

    if ((err == SDC_OK) && (internal_out_len != 0)) {
        /* allocate output buffer */
        internal_out = malloc(internal_out_len);
        if (!internal_out)
            err = SDC_NO_MEM;
    }

    if (err == SDC_OK) {
        err = sdc_unwrap_selector(session,
                                  type,
                                  &internal_desc,
                                  in_data,
                                  in_len,
                                  internal_out,
                                  &internal_out_len,
                                  aad_data, aad_len,
                                  tag_data, tag_len,
                                  iv, iv_len);
    }

    if (err == SDC_OK) {
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        if (internal_out)
            free(internal_out);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_wrap_formatted_extended(sdc_session_t *session,
                                        const sdc_wrap_unwrap_type_t *type,
                                        const uint8_t *in_data, const size_t in_len,
                                        const uint8_t *iv, const size_t iv_len,
                                        const uint8_t *aad_data, const size_t aad_len,
                                        const size_t tag_len,
                                        uint8_t **formatted_data, size_t *formatted_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_wrap_unwrap_desc_t internal_desc;
    size_t internal_iv_len;
    bool explicit_iv;
    size_t internal_tag_len;
    sdc_form_header_generic_t form_header;
    sdc_form_header_wrap_unwrap_generic_t *form_wrap;
    uint8_t *internal_formatted_buffer = NULL;
    uint8_t *iv_tmp;
    size_t internal_out_len = 0;

    if (SDC_OK != sdc_intern_check_data_input_buffer(aad_data, aad_len))
        err = SDC_AAD_DATA_INVALID;

    /* if the iv is externally determined the len has to be set externally too */
    explicit_iv = (iv != NULL);
    if (explicit_iv && (iv_len == SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    err = sdc_wrap_unwrap_common_checks_defaults(err,
                                                 session,
                                                 type, &internal_desc,
                                                 in_data, in_len,
                                                 formatted_data, formatted_len, SDC_FORMATTED_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    /* get defaults if not set by application */
    internal_iv_len = iv_len;
    if (internal_iv_len == SDC_IV_USE_DEFAULT) {
        internal_iv_len = internal_desc.iv.dflt;
    }
    internal_tag_len = tag_len;
    if (internal_tag_len == SDC_TAG_USE_DEFAULT) {
        internal_tag_len = internal_desc.tag.dflt;
    }

    err = sdc_wrap_unwrap_common_check_iv_tag_in_length(
        &internal_desc,
        true,
        in_len,
        internal_iv_len,
        internal_tag_len);

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_wrap_unwrap_type_get_opt_bmsk(type, &opt_bmsk);

        /* using formatted with additional options is not allowed */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_OP_NOT_SUPPORTED;
    }

    if (err == SDC_OK) {
        /* determine the output length */
        err = sdc_arch_wrap_get_out_len(session, type, &internal_desc, in_len, &internal_out_len);
    }

    if (err == SDC_OK) {
        sdc_inter_form_header_init(&form_header);

        err = sdc_intern_wrap_unwrap_formatted_fill_header (session,
                                                            type,
                                                            internal_out_len,
                                                            internal_iv_len,
                                                            internal_tag_len,
                                                            aad_len,
                                                            &form_header);

        if (err == SDC_OK) {
            internal_formatted_buffer = malloc(form_header.overall_formatted_len);
            if (internal_formatted_buffer == NULL)
                err = SDC_NO_MEM;
        }

        if (err == SDC_OK) {
            err = sdc_intern_wrap_unwrap_formatted_update_pointers(&form_header,
                                                                   internal_formatted_buffer);
        }

        if (err == SDC_OK) {
            form_wrap = &form_header.wrap_unwrap;
            err = sdc_intern_formatted_write_header(&form_header,
                                                    internal_formatted_buffer);

            if (err == SDC_OK) {
                if (internal_iv_len != 0) {
                    if (explicit_iv) {
                        /* copy explicitly specified iv */
                        memcpy (form_wrap->iv, iv, iv_len);
                    } else {
                        iv_tmp = NULL;
                        err = sdc_random_gen_buffer(session, internal_iv_len, &iv_tmp);
                        if (err == SDC_OK) {
                            memcpy (form_wrap->iv, iv_tmp, internal_iv_len);
                        }
                        if (iv_tmp != NULL)
                            free(iv_tmp);
                    }
                }
            }
            if ((err == SDC_OK) && (aad_len != 0)) {
                /* copy aad_data */
                memcpy (form_wrap->aad, aad_data, aad_len);
            }

            if (err == SDC_OK) {
                err = sdc_wrap_selector(session,
                                        type,
                                        &internal_desc,
                                        in_data, in_len,
                                        form_wrap->data, form_wrap->data_len,
                                        form_wrap->aad, form_wrap->aad_len,
                                        form_wrap->iv, form_wrap->iv_len,
                                        form_wrap->tag, form_wrap->tag_len);
            }
        }

        if (err == SDC_OK) {
            *formatted_data = internal_formatted_buffer;
            *formatted_len = form_header.overall_formatted_len;
        } else {
            if (internal_formatted_buffer)
                free (internal_formatted_buffer);
        }

        sdc_inter_form_header_free(&form_header);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_wrap_formatted(sdc_session_t *session,
                               const sdc_wrap_unwrap_type_t *type,
                               const uint8_t *in_data, const size_t in_len,
                               uint8_t **formatted_data, size_t *formatted_len)
{
    return sdc_wrap_formatted_extended(session,
                                       type,
                                       in_data, in_len,
                                       NULL, SDC_IV_USE_DEFAULT,
                                       NULL, 0,
                                       SDC_TAG_USE_DEFAULT,
                                       formatted_data, formatted_len);
}

sdc_error_t sdc_unwrap_formatted(sdc_session_t *session,
                                 const sdc_wrap_unwrap_type_t *type,
                                 const uint8_t *formatted_data, const size_t formatted_len,
                                 uint8_t **out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_form_header_generic_t form_header;
    sdc_form_header_wrap_unwrap_generic_t *form_wrap = NULL;
    sdc_wrap_unwrap_desc_t desc;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;
    sdc_wrap_unwrap_alg_t alg;
    sdc_wrap_unwrap_blk_t blk;

    if (SDC_OK != sdc_intern_check_init_data_output_buffer(out_data, out_len))
        err = SDC_OUT_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        err = SDC_FORMATTED_DATA_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err != SDC_OK)
        return err;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_WRAP_UNWRAP)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        form_wrap = &form_header.wrap_unwrap;
        /* we need to typecast as the same pointers in wrap struct are used for
         * read and write. Before calling unwrap we will make it const again */
        err = sdc_intern_wrap_unwrap_formatted_update_pointers(
            &form_header,
            (uint8_t*)formatted_data);

        /* check if given type matches the one from the formatted data */
        if (err == SDC_OK) {
            err = sdc_wrap_unwrap_type_get_algorithm(type, &alg);

            if ((err == SDC_OK) && (alg != form_wrap->alg))
                err = SDC_ALG_MODE_INVALID;
        }
        if (err == SDC_OK) {
            err = sdc_wrap_unwrap_type_get_block_mode(type, &blk);

            if ((err == SDC_OK) && (blk != form_wrap->blk))
                err = SDC_ALG_MODE_INVALID;
        }

        if (err == SDC_OK) {
            uint64_t opt_bmsk;

            err = sdc_wrap_unwrap_type_get_opt_bmsk(type, &opt_bmsk);

            /* using formatted with additional options is not allowed */
            if ((err == SDC_OK) && (opt_bmsk != 0))
                err = SDC_OP_NOT_SUPPORTED;
        }

        if (err == SDC_OK) {
            err = sdc_wrap_unwrap_desc_fill (session, type, &desc, form_header.wrap_unwrap.data_len);

            if (err == SDC_OK) {
                err = sdc_wrap_unwrap_common_check_iv_tag_in_length(
                    &desc,
                    false,
                    form_wrap->data_len,
                    form_wrap->iv_len,
                    form_wrap->tag_len);

                if (err == SDC_OK) {
                    /* determine the max output length */
                    err = sdc_arch_unwrap_get_max_out_len(
                        session, type, &desc,
                        form_wrap->data_len,
                        &internal_out_len);
                }

                if ((err == SDC_OK) && (internal_out_len != 0)) {
                    /* allocate output buffer */
                    internal_out = malloc(internal_out_len);
                    if (!internal_out)
                        err = SDC_NO_MEM;
                }

                if (err == SDC_OK) {
                    err = sdc_unwrap_selector(session,
                                              type,
                                              &desc,
                                              (const uint8_t*)form_wrap->data,
                                              form_wrap->data_len,
                                              internal_out,
                                              &internal_out_len,
                                              (const uint8_t*)form_wrap->aad, form_wrap->aad_len,
                                              (const uint8_t*)form_wrap->tag, form_wrap->tag_len,
                                              (const uint8_t*)form_wrap->iv, form_wrap->iv_len);
                }
            }
        }
    }

    if (err == SDC_OK) {
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        if (internal_out)
            free(internal_out);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len)
{
    return sdc_intern_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               secret_mod_data, secret_mod_len);
}

sdc_error_t sdc_unwrap_formatted_autoload_key(sdc_session_t *session,
                                              const uint8_t *formatted_data, const size_t formatted_len)
{
    return sdc_unwrap_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               NULL, 0);
}

/* defined in sdc_op_conv.h */
sdc_error_t sdc_unwrap_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_wrap_unwrap_type_t **type)
{
    sdc_error_t err = SDC_OK;
    sdc_form_header_generic_t form_header;
    sdc_wrap_unwrap_type_t *tmp = NULL;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        return SDC_FORMATTED_DATA_INVALID;

    if (!type) {
        return SDC_ALG_MODE_INVALID;
    }

    // initialize
    *type = NULL;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_WRAP_UNWRAP)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_type_alloc(&tmp);
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_type_set_algorithm(tmp, form_header.wrap_unwrap.alg);
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_type_set_block_mode(tmp, form_header.wrap_unwrap.blk);
    }

    if (err == SDC_OK) {
        *type = tmp;
    } else {
        if (tmp != NULL)
            sdc_wrap_unwrap_type_free(tmp);
    }

    return err;
}

/* defined in sdc_op_common.h */
const sdc_wrap_unwrap_type_t *sdc_wrap_unwrap_get_default(void)
{
    return sdc_arch_wrap_unwrap_get_default();
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_alloc(sdc_wrap_unwrap_type_t **type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_wrap_unwrap_type_alloc(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_free(sdc_wrap_unwrap_type_t *type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_wrap_unwrap_type_free(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_set_algorithm(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t alg)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((alg < SDC_WRAP_ALG_FIRST) || (alg >= SDC_WRAP_ALG_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_wrap_unwrap_type_set_algorithm(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_set_block_mode(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t blk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((blk < SDC_WRAP_BLK_FIRST) || (blk >= SDC_WRAP_BLK_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_wrap_unwrap_type_set_block_mode(type, blk);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_set_opt_bmsk(sdc_wrap_unwrap_type_t *type, uint64_t opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (opt_bmsk != 0) /* nothing supported so far */
        return SDC_OP_NOT_SUPPORTED;

    /* as we don't support any option at the moment, we don't need any arch interface */
    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_get_algorithm(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t *alg)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!alg)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_wrap_unwrap_type_get_algorithm(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_get_block_mode(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t *blk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!blk)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_wrap_unwrap_type_get_block_mode(type, blk);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_type_get_opt_bmsk(const sdc_wrap_unwrap_type_t *type, uint64_t *opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!opt_bmsk)
        return SDC_INVALID_PARAMETER;

    /* as we don't support any option at the moment, we don't need any arch interface - simply return 0*/
    *opt_bmsk = 0;
    return SDC_OK;
}


/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_alloc(sdc_wrap_unwrap_desc_t **desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *desc = malloc(sizeof(sdc_wrap_unwrap_desc_t));
    if (*desc == NULL) {
        return SDC_NO_MEM;
    }

    memset(*desc, 0, sizeof(sdc_wrap_unwrap_desc_t));

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_free(sdc_wrap_unwrap_desc_t *desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    free(desc);

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_fill (
    sdc_session_t *session,
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *desc,
    size_t total_data_len)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_wrap_unwrap_desc_fill(session, type, desc, total_data_len);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_get_max_chunk_len(const sdc_wrap_unwrap_desc_t *desc,
                                                       size_t *max_val)
{
    size_t max_chunk;

    if (!desc || !max_val)
        return SDC_INVALID_PARAMETER;

    max_chunk = desc->data.max_chunk_len;
    if (desc->data.chunk_len_aligned) {
        max_chunk -= (max_chunk % desc->data.block_len);
    }

    if (max_chunk == 0)
        return SDC_INTERNAL_ERROR;

    *max_val = max_chunk;

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_get_tag(sdc_wrap_unwrap_desc_t *desc,
                                         size_t *min_val,
                                         size_t *max_val,
                                         size_t *mod,
                                         size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->tag),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_wrap_unwrap_desc_get_iv(sdc_wrap_unwrap_desc_t *desc,
                                        size_t *min_val,
                                        size_t *max_val,
                                        size_t *mod,
                                        size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->iv),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

const char* sdc_wrap_unwrap_algorithm_name (sdc_wrap_unwrap_alg_t alg)
{
    size_t idx;
    size_t elems;

    if ((alg < SDC_WRAP_ALG_FIRST) || (alg >= SDC_WRAP_ALG_END))
        return NULL;

    idx = alg;
    elems = sizeof(sdc_wrap_unwrap_alg_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_wrap_unwrap_alg_names[idx];
    }

    return NULL;
}

const char* sdc_wrap_unwrap_block_mode_name (sdc_wrap_unwrap_blk_t blk)
{
    size_t idx;
    size_t elems;

    if ((blk < SDC_WRAP_BLK_FIRST) || (blk >= SDC_WRAP_BLK_END))
        return NULL;

    idx = blk;
    elems = sizeof(sdc_wrap_unwrap_blk_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_wrap_unwrap_blk_names[idx];
    }

    return NULL;
}

sdc_error_t sdc_wrap_unwrap_get_key_key_lens_fmt(
    sdc_wrap_unwrap_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len)
{
    sdc_key_desc_t key_desc;
    sdc_error_t err;

    if (!type)
        return SDC_ALG_MODE_INVALID;

    err = sdc_arch_wrap_unwrap_key_desc_fill(type, &key_desc);

    if (err == SDC_OK)
        err = sdc_intern_key_lens_fmt(&key_desc,
                                      sup_key_fmt_protect, sup_key_fmt_unprotect,
                                      sup_key_lens, dflt_key_len);

    return err;
}

sdc_error_t sdc_wrap_init(sdc_session_t *session,
                          const sdc_wrap_unwrap_type_t *type,
                          const sdc_wrap_unwrap_desc_t *desc,
                          const size_t total_inlen,
                          const size_t total_aadlen,
                          size_t *tag_len,
                          uint8_t **iv, size_t *iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;
    bool explicit_iv = false;
    uint8_t *internal_iv = NULL;
    size_t internal_iv_len = 0;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_WRAP, SDC_NO_OP);

    if(err == SDC_OK)
        err = sdc_get_wrap_unwrap_tag_length(desc, tag_len);

    if (err == SDC_OK) {
        if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv)) {
            err = SDC_IV_INVALID;
        }
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            if (internal_iv_len == SDC_IV_USE_DEFAULT) {
                internal_iv_len = desc->iv.dflt;
            }
            internal_iv = NULL;
            /* there might be formats without IV */
            if (internal_iv_len != 0) {
                err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
            }
        } else {
            internal_iv = *iv;
            internal_iv_len = *iv_len;
        }
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_common_check_iv(desc, internal_iv_len);
    }

    if (err == SDC_OK) {
        err = sdc_common_wrapunwrap_init(session, type, desc, total_inlen, total_aadlen, *tag_len,
                                         internal_iv, internal_iv_len, true);
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap_init(sdc_session_t *session,
                            const sdc_wrap_unwrap_type_t *type,
                            const sdc_wrap_unwrap_desc_t *desc,
                            const size_t total_inlen,
                            const size_t total_aadlen,
                            size_t tag_len,
                            const uint8_t *iv, size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_UNWRAP, SDC_NO_OP);

    if(err == SDC_OK) {
        if (sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT) != SDC_OK) {
            err = SDC_IV_INVALID;
        }
    }

    if(err == SDC_OK)
        err = sdc_wrap_unwrap_common_check_iv_tag_in_length(desc, false, total_inlen, iv_len, tag_len);

    if (err == SDC_OK) {
        err = sdc_common_wrapunwrap_init(session, type, desc, total_inlen, total_aadlen, tag_len,
                                         iv, iv_len, false);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }
    return err;
}

sdc_error_t sdc_wrap_aad_update(sdc_session_t *session,
                                const sdc_wrap_unwrap_type_t *type,
                                const sdc_wrap_unwrap_desc_t *desc,
                                const uint8_t *aad_data, const size_t aad_len)
{
    sdc_error_t err =  SDC_OK;
    sdc_error_t tmp_err;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_WRAP, SDC_OP_INITILIZED);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(aad_data, aad_len) != SDC_OK)
            err = SDC_AAD_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_wrapunwrap_aad_update(session, type, desc,
                                               aad_data, aad_len, true);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap_aad_update(sdc_session_t *session,
                                  const sdc_wrap_unwrap_type_t *type,
                                  const sdc_wrap_unwrap_desc_t *desc,
                                  const uint8_t *aad_data, const size_t aad_len)
{
    sdc_error_t err =  SDC_OK;
    sdc_error_t tmp_err;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(aad_data, aad_len) != SDC_OK)
            err = SDC_AAD_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_wrapunwrap_aad_update(session, type, desc,
                                               aad_data, aad_len, false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_wrap_update(sdc_session_t *session,
                            const sdc_wrap_unwrap_type_t *type,
                            const sdc_wrap_unwrap_desc_t *desc,
                            const uint8_t *in_data, const size_t in_len,
                            uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_WRAP, SDC_OP_INITILIZED);

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_check_output(out_data, out_len);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
            err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_wrapunwrap_update(session, type, desc,
                                           in_data, in_len,
                                           out_data, out_len,
                                           true);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap_update(sdc_session_t *session,
                              const sdc_wrap_unwrap_type_t *type,
                              const sdc_wrap_unwrap_desc_t *desc,
                              const uint8_t *in_data, const size_t in_len,
                              uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_wrap_unwrap_check_output(out_data, out_len);


    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
        err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_wrapunwrap_update(session, type, desc,
                                           in_data, in_len,
                                           out_data, out_len,
                                           false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }
    return err;
}

sdc_error_t sdc_wrap_finalize(sdc_session_t *session,
                              const sdc_wrap_unwrap_type_t *type,
                              const sdc_wrap_unwrap_desc_t *desc,
                              uint8_t *out_data, size_t *out_len,
                              uint8_t **tag_data, size_t tag_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;
    uint8_t *internal_tag =  NULL;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_WRAP, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_wrap_unwrap_check_output(out_data, out_len);

    if(err == SDC_OK) {
        err = sdc_wrap_unwrap_common_check_tag(desc, tag_len);
    }

    /* Application provided tag_len in FINALIZE and output tag_len of INIT should be same */
    if((err == SDC_OK) && (tag_len != session->iuf_wrap_unwrap.tag_len))
        err = SDC_TAG_DATA_INVALID;

    if (err == SDC_OK) {
        if (tag_len != 0) {
            internal_tag = malloc(tag_len);
            if (!internal_tag)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK)
        err = sdc_common_wrapunwrap_finalize(session, type, desc,
                                             internal_tag, tag_len,
                                             NULL, 0,
                                             out_data, out_len,
                                             true);

    if ((err == SDC_OK) && (tag_data)) {
        *tag_data = internal_tag;
    } else {
        /* clean allocated memory */
        free(internal_tag);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_WRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_unwrap_finalize(sdc_session_t *session,
                                const sdc_wrap_unwrap_type_t *type,
                                const sdc_wrap_unwrap_desc_t *desc,
                                uint8_t *out_data, size_t *out_len,
                                const uint8_t *tag_data, size_t tag_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_UNWRAP, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_wrap_unwrap_check_output(out_data, out_len);

    if (err == SDC_OK) {
        if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(tag_data, tag_len,
                                                           SDC_TAG_USE_DEFAULT))
            err = SDC_TAG_DATA_INVALID;
    }

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_common_check_tag(desc, tag_len);
    }

    /* Application provided tag_len in INIT and FINALIZE should be same */
    if((err == SDC_OK) && (tag_len != session->iuf_wrap_unwrap.tag_len))
        err = SDC_TAG_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_common_wrapunwrap_finalize(session, type, desc,
                                             NULL, 0,
                                             tag_data, tag_len,
                                             out_data, out_len,
                                             false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_UNWRAP, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}
